//
//  TAWDevicesViewController.swift
//  TAW
//
//  Created by Andrew Steven on 25/02/2019.
//  Copyright © 2019 PixelBeard. All rights reserved.
//

import UIKit
import Spruce

class TAWDevicesViewController: PBAnimatedViewController {
    
    @IBOutlet weak var tableView: UITableView!
    
    var devices: [Device] = []
    var isUnlinkingDevice = false
    private let pullToRefresh = UIRefreshControl()

    override func viewDidLoad() {
        super.viewDidLoad()
        
        self.tableView.refreshControl = pullToRefresh
        self.tableView.tableFooterView = UIView()
        
        pullToRefresh.addTarget(self, action: #selector(getMyDevices), for: .valueChanged)
        pullToRefresh.tintColor = TAW.Colors.pullToRefreshColor
        pullToRefresh.attributedTitle =
            NSAttributedString(string: "LOADING",
                               attributes: [NSAttributedString.Key.foregroundColor: TAW.Colors.pullToRefreshColor,
                                            NSAttributedString.Key.font: TAW.Fonts.dinAlternate10!])
    }
    
    override func viewWillAppear(_ animated: Bool) {
        self.animations =
            self.isMovingToParent ? [.slide(.left, .slightly), .fadeIn] : [.slide(.right, .slightly), .fadeIn]
        super.viewWillAppear(animated)
        self.getMyDevices()
    }
    
    // MARK: - Button actions -
    
    @IBAction func goBack() {
        self.navigationController?.popViewController(animated: true)
    }

    // MARK: - API calls -
    
    @objc private func getMyDevices() {
        
        if !isUnlinkingDevice {
            self.pullToRefresh.endRefreshing()
            PBLoadingOverlay.shared.showOverlay(view: (self.navigationController?.view)!, withTitle: "Fetching devices")
            
            APIClient.getDevices { (jsonResponse, _) in
                if jsonResponse["success"] as? Bool == true {
                    if let devicesJson = jsonResponse["devices"] as? [[String: Any]] {
                        self.devices = []
                        self.tableView.reloadData()
                        for deviceJson in devicesJson {
                            if let device = Device(JSON: deviceJson) {
                                self.devices.append(device)
                            }
                        }
                        
                        self.devices = self.devices.sorted {$0.lastLoggedIn! > $1.lastLoggedIn!}
                        if let index: Int = self.devices.firstIndex(where: {$0.deviceIdentifier == LocalStorage.shared.uuid}) {
                            let current = self.devices[index]
                            self.devices.remove(at: index)
                            self.devices.insert(current, at: 0)
                        }
                        
                        PBLoadingOverlay.shared.hideOverlay {
                            self.tableView.reloadData()
                            self.animateTableView(self.tableView)
                        }
                    }
                } else {
                    PBLoadingOverlay.shared.hideOverlay {
                        let message = jsonResponse["message"] as? String ?? ""
                        if message == "Invalid Token" {
                            self.invalidateSession()
                        } else {
                            self.showAlertForAPIError(message)
                        }
                    }
                }
            }
        } else {
            self.tableView.reloadData()
            self.animateTableView(self.tableView)
        }
    }
    
    private func unlinkDevice(_ device: Device) {
        
        PBLoadingOverlay.shared.showOverlay(view: (self.navigationController?.view)!, withTitle: "Unlinking device")
        
        let params = [
            "id": device.id
        ]
        
        APIClient.unlinkDevice(parameters: params) { (jsonResponse, _) in
            
            let message = jsonResponse["message"] as? String ?? ""
            
            if jsonResponse["success"] as? Bool == true {
                PBLoadingOverlay.shared.hideOverlay {
                    if let index = self.devices.firstIndex(of: device) {
                        self.devices.remove(at: index)
                        self.tableView.deleteRows(at: [IndexPath(row: index, section: 0)], with: .automatic)
                    }
                    if self.isUnlinkingDevice {
                        self.goBackToLogin(message)
                    } else {
                        self.showAlertWithTitle("Success", message: message)
                    }
                }
            } else {
                PBLoadingOverlay.shared.hideOverlay {
                    if message == "Invalid Token" {
                        self.invalidateSession()
                    } else {
                        self.showAlertForAPIError(message)
                    }
                }
            }
        }
    }
    
    private func goBackToLogin(_ message: String) {
        self.performSegue(withIdentifier: "unwindToLoginView", sender: nil)
    }
}

// MARK: - UITableViewDelegate & UITableViewDataSource -

extension TAWDevicesViewController: UITableViewDelegate, UITableViewDataSource {
    func numberOfSections(in tableView: UITableView) -> Int {
        return 1
    }
    
    func tableView(_ tableView: UITableView,
                   numberOfRowsInSection section: Int) -> Int {
        return devices.count
    }
    
    func tableView(_ tableView: UITableView,
                   willDisplay cell: UITableViewCell,
                   forRowAt indexPath: IndexPath) {
        
        cell.spruce.prepare(with: self.tableViewAnimations)
        var animation = SpringAnimation(duration: 0.7)
        animation.animationOptions = [UIView.AnimationOptions.allowUserInteraction]
        
        cell.spruce.animate(self.tableViewAnimations,
                            animationType: animation,
                            sortFunction: self.tableViewSortFunction!)
        cell.layoutSubviews()
    }
    
    func tableView(_ tableView: UITableView,
                   cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let cell: TAWDevicesTableViewCell =
            tableView.dequeueReusableCell(withIdentifier: "myDevicesCell",
                                          for: indexPath) as! TAWDevicesTableViewCell
        cell.setDevice(self.devices[indexPath.row])
        cell.delegate = self
        return cell
    }
    
    func tableView(_ tableView: UITableView, estimatedHeightForRowAt indexPath: IndexPath) -> CGFloat {
        return UITableView.automaticDimension
    }
    
    func tableView(_ tableView: UITableView, canEditRowAt indexPath: IndexPath) -> Bool {
        let device = self.devices[indexPath.row]
        return device.deviceIdentifier != LocalStorage.shared.uuid
    }
    
    func tableView(_ tableView: UITableView,
                   commit editingStyle: UITableViewCell.EditingStyle,
                   forRowAt indexPath: IndexPath) {
        if editingStyle == .delete {
            
            let device = self.devices[indexPath.row]
            self.showUnlinkAlert(device)
        }
    }
    
    func showUnlinkAlert(_ device: Device) {
        
        let alert = UIAlertController(title: "Are you sure?",
                                      message: "Are you sure you want to unlink this device from your account? Devices can only be linked / unlinked once every 30 days",
                                      preferredStyle: .alert)
        alert.addAction(UIAlertAction(title: "NO", style: .cancel, handler: nil))
        alert.addAction(UIAlertAction(title: "YES", style: .destructive, handler: { (_) in
            self.unlinkDevice(device)
        }))
        self.navigationController?.present(alert, animated: true, completion: nil)
    }
}

extension TAWDevicesViewController: TAWDevicesTableViewCellDelegate {
    func deleteDevice(_ device: Device) {
        self.showUnlinkAlert(device)
    }
}
